/* SPDX-License-Identifier: GPL-2.0-only */
#ifndef LINUX_DRIVER_H_
#define LINUX_DRIVER_H_

#include <driver.h>
#include <linux/slab.h>
#include <linux/bug.h>
#include <mmu.h>

#define device_driver driver

#define __devm_wrapper(fn, dev, ...) ({ BUG_ON(!dev); fn(__VA_ARGS__); })

#define devm_kmalloc(...)		__devm_wrapper(kmalloc, __VA_ARGS__)
#define devm_kvasprintf(...)		__devm_wrapper(kvasprintf, __VA_ARGS__)
#define devm_kasprintf(...)		__devm_wrapper(kasprintf, __VA_ARGS__)
#define devm_kzalloc(...)		__devm_wrapper(kzalloc, __VA_ARGS__)
#define devm_kmalloc_array(...)		__devm_wrapper(kmalloc_array, __VA_ARGS__)
#define devm_kcalloc(...)		__devm_wrapper(kcalloc, __VA_ARGS__)
#define devm_kfree(...)			__devm_wrapper(kfree, __VA_ARGS__)
#define devm_kstrdup(...)		__devm_wrapper(kstrdup, __VA_ARGS__)
#define devm_kstrdup_const(...)		__devm_wrapper(kstrdup_const, __VA_ARGS__)
#define devm_kmemdup(...)		__devm_wrapper(kmemdup, __VA_ARGS__)
#define devm_bitmap_zalloc(dev, nbits, gfp)	\
	__devm_wrapper(bitmap_zalloc, dev, nbits)

#define device_register register_device
#define device_unregister unregister_device

#define driver_register register_driver
#define driver_unregister unregister_driver


static inline void __iomem *dev_platform_ioremap_resource(struct device *dev,
							  int resource)
{
	/*
	 * barebox maps everything outside the RAM banks suitably for MMIO,
	 * so we don't need to do anything besides requesting the regions
	 * and can leave the memory attributes unchanged.
	 */
	return dev_request_mem_region(dev, resource);
}

static inline void __iomem *devm_ioremap(struct device *dev,
					 resource_size_t start,
					 resource_size_t size)
{
	if (start)
		remap_range((void *)start, size, MAP_UNCACHED);

	return IOMEM(start);
}

/**
 * dev_set_drvdata - set driver private data for device
 * @dev: device instance
 * @data: driver-specific data
 *
 * Returns private driver data or NULL if none was set.
 *
 * NOTE: This does _not_ return the match data associated with
 * the match. For that use device_get_match_data instead.
 */
static inline void dev_set_drvdata(struct device *dev, void *data)
{
	dev->driver_data = data;
}

/**
 * dev_get_drvdata - get driver match data associated for device
 * @dev: device instance
 * @data: driver-specific data
 *
 * Set some driver and device specific data for later retrieval
 * by dev_get_drvdata.
 *
 * NOTE: This does _not_ return the match data associated with
 * the match. For that use device_get_match_data instead.
 */
static inline void *dev_get_drvdata(const struct device *dev)
{
	return dev->driver_data;
}

#endif
